<?php

namespace Database\Seeders;

use Illuminate\Database\Seeder;
use App\Models\Inventory;
use App\Models\Brand;
use App\Models\Category;
use App\Models\Supplier;
use App\Models\User;
use App\Models\Setting;
use App\Models\Purchase;
use App\Models\PurchaseItem;
use App\Models\InventoryItem;
use Illuminate\Support\Str;
use Faker\Factory as Faker;

class InventorySeeder extends Seeder
{
    /**
     * Run the database seeds.
     */
    public function run(): void
    {
        $faker = Faker::create();

        $brands = Brand::pluck('id')->toArray();
        $categories = Category::pluck('id')->toArray();
        $suppliers = Supplier::pluck('id')->toArray();

        $colors = ['Black', 'White', 'Blue', 'Red', 'Green', 'Gray', 'Gold'];
        $storageOptions = ['32GB', '64GB', '128GB', '256GB', '512GB'];
        $ramOptions = ['2GB', '3GB', '4GB', '6GB', '8GB', '12GB'];
        $conditions = ['new', 'refurbished', 'used'];
        $statuses = ['available', 'out_of_stock', 'discontinued'];
        $processors = ['Snapdragon', 'MediaTek Helio', 'Apple A-Series', 'Exynos'];
        $os = ['Android', 'iOS', 'HarmonyOS'];

        /*
        |--------------------------------------------------------------------------
        | 1) INVENTORY (20 PRODUCTS)
        |--------------------------------------------------------------------------
        */
        for ($i = 1; $i <= 10; $i++) {

            $brand_id = $faker->randomElement($brands);
            $category_id = $faker->randomElement($categories);

            // ----- PRICE LOGIC (ROUND THOUSANDS + PROFIT) -----

            // Base purchase price in thousands (e.g. 20–200 => 20,000–200,000)
            $purchaseThousands = $faker->numberBetween(20, 200); // adjust range if needed
            $purchase_price = $purchaseThousands * 1000;

            // Profit margin between 5% and 30%
            $marginPercent = $faker->numberBetween(5, 30);
            $rawSelling = $purchase_price * (1 + $marginPercent / 100);

            // Round selling price to nearest thousand
            $selling_price = round($rawSelling / 1000) * 1000;

            // Ensure selling price > purchase price
            if ($selling_price <= $purchase_price) {
                $selling_price = $purchase_price + 1000;
            }

            // Wholesale price between purchase and selling, closer to purchase
            $wholesaleMin = $purchase_price + 1000; // at least some profit over purchase
            $wholesaleMax = $selling_price - 1000;  // slightly cheaper than selling

            if ($wholesaleMax <= $wholesaleMin) {
                $wholesale_price = $purchase_price + 1000;
            } else {
                $wholesaleThousands = $faker->numberBetween(
                    (int)($wholesaleMin / 1000),
                    (int)($wholesaleMax / 1000)
                );

                $wholesale_price = $wholesaleThousands * 1000;
            }

            // --------------------------------------------------

            Inventory::create([
                'sku' => strtoupper(Str::random(10)),
                'brand_id' => $brand_id,
                'category_id' => $category_id,
                'model' => $faker->unique()->bothify('Model ###'),
                'color' => $faker->randomElement($colors),
                'storage' => $faker->randomElement($storageOptions),
                'ram' => $faker->randomElement($ramOptions),

                'purchase_price' => $purchase_price,
                'selling_price' => $selling_price,
                'wholesale_price' => $wholesale_price,

                'minimum_stock' => $faker->numberBetween(3, 10),
                'description' => $faker->sentence(12),
                'condition' => $faker->randomElement($conditions),
                'warranty_period' => $faker->randomElement(['6 Months', '1 Year', 'No Warranty']),
                'camera' => $faker->numberBetween(8, 108) . ' MP',
                'battery' => $faker->numberBetween(2500, 6000) . ' mAh',
                'screen_size' => $faker->randomElement(['5.5"', '6.1"', '6.5"', '6.7"']),
                'processor' => $faker->randomElement($processors),
                'operating_system' => $faker->randomElement($os),
                'supplier_id' => $faker->optional()->randomElement($suppliers),
                'status' => $faker->randomElement($statuses),
                'is_active' => true,

                // if you have quantity column in inventory table:
                // 'quantity' => 0,
            ]);
        }

        echo "✅ 20 inventory items added with realistic, profitable pricing!\n";

        /*
        |--------------------------------------------------------------------------
        | 2) PURCHASES, PURCHASE ITEMS & INVENTORY ITEMS (IMEI LEVEL)
        |--------------------------------------------------------------------------
        */

        $inventories = Inventory::all();
        if ($inventories->count() === 0 || empty($suppliers)) {
            echo "⚠️ Skipping purchases: no inventory or suppliers found.\n";
            return;
        }

        // Get admin user for created_by
        $adminId = User::where('role', 'admin')->value('id') ?? User::first()->id;

        // Purchase number prefix from settings (fallback PUR)
        $prefix = Setting::where('key', 'purchase_prefix')->value('value') ?? 'PUR';
        $lastPurchaseId = Purchase::max('id') ?? 0;

        $numberOfPurchases = 500; // you can increase if you want

        for ($p = 1; $p <= $numberOfPurchases; $p++) {

            $purchaseNumber = $prefix . '-' . str_pad($lastPurchaseId + $p, 4, '0', STR_PAD_LEFT);
            $purchaseDate = now()->subDays($faker->numberBetween(1, 30))->toDateString();
            $supplierId = $faker->randomElement($suppliers);

            // Choose 2–4 random inventory products for this purchase
            $itemsForPurchase = $inventories->random($faker->numberBetween(2, 4));

            // Calculate subtotal from items
            $subtotal = 0;
            $itemsData = [];

            foreach ($itemsForPurchase as $inv) {
                $qty = $faker->numberBetween(1, 5); // quantity per line
                $unitPrice = $inv->purchase_price; // cost per unit = inventory purchase_price
                $lineSubtotal = $qty * $unitPrice;
                $subtotal += $lineSubtotal;

                $itemsData[] = [
                    'inventory'     => $inv,
                    'quantity'      => $qty,
                    'unit_price'    => $unitPrice,
                    'line_subtotal' => $lineSubtotal,
                ];
            }

            $taxAmount = 0; // keep simple
            $discount = 0;
            $totalAmount = $subtotal + $taxAmount - $discount;

            // Create Purchase
            $purchase = Purchase::create([
                'purchase_number'  => $purchaseNumber,
                'supplier_id'      => $supplierId,
                'created_by'       => $adminId,
                'purchase_date'    => $purchaseDate,
                'subtotal'         => $subtotal,
                'tax_amount'       => $taxAmount,
                'discount'         => $discount,
                'total_amount'     => $totalAmount,
                'paid_amount'      => 0,
                'due_amount'       => $totalAmount,
                'payment_status'   => 'pending',
                'status'           => 'completed',
                'reference_number' => 'REF-' . Str::upper(Str::random(6)),
                'notes'            => $faker->sentence(8),
            ]);

            // Create PurchaseItems & InventoryItems (per-device IMEIs)
            foreach ($itemsData as $item) {
                /** @var \App\Models\Inventory $inventory */
                $inventory = $item['inventory'];
                $qty = $item['quantity'];
                $unitPrice = $item['unit_price'];
                $lineSubtotal = $item['line_subtotal'];

                // Purchase item (is_device = 1 for mobiles)
                $purchaseItem = PurchaseItem::create([
                    'purchase_id'  => $purchase->id,
                    'inventory_id' => $inventory->id,
                    'quantity'     => $qty,
                    'unit_price'   => $unitPrice,
                    'subtotal'     => $lineSubtotal,
                    'is_device'    => 1,
                ]);

                // Update aggregate inventory quantity & status
                // if (property_exists($inventory, 'quantity') || isset($inventory->quantity)) {
                //     $inventory->quantity = ($inventory->quantity ?? 0) + $qty;

                //     if ($inventory->quantity > 0 && $inventory->status === 'out_of_stock') {
                //         $inventory->status = 'available';
                //     }
                // }

                // Keep purchase_price in sync
                $inventory->purchase_price = $unitPrice;
                $inventory->save();

                // Create per-device InventoryItem rows with fake IMEIs
                for ($iDev = 1; $iDev <= $qty; $iDev++) {
                    $imei1 = $faker->numerify('###############'); // 15 chars (mix digits/letters to avoid int limits)
                    $imei2 = $faker->numerify('###############');

                    InventoryItem::create([
                        'inventory_id'         => $inventory->id,
                        'purchase_item_id'     => $purchaseItem->id,
                        'imei_1'               => $imei1,
                        'imei_2'               => $imei2,
                        'serial_number'        => strtoupper(Str::random(12)),

                        'color'                => $inventory->color,
                        'storage'              => $inventory->storage,
                        'ram'                  => $inventory->ram,

                        'purchase_price'       => $unitPrice,
                        'selling_price'        => $inventory->selling_price,

                        'supplier_id'          => $supplierId,
                        'purchase_date'        => $purchaseDate,

                        'condition'            => $inventory->condition ?? 'new',
                        'warranty_months'      => $faker->randomElement([0, 3, 6, 12]),
                        'warranty_expiry_date' => null, // you can calculate from date if you want

                        'status'               => 'available',
                        'location'             => $faker->randomElement(['Main Store', 'Branch 1', 'Warehouse']),

                        'sold_date'            => null,
                        'customer_id'          => null,

                        'notes'                => $faker->boolean(20) ? $faker->sentence(6) : null,
                        'is_active'            => true,
                    ]);
                }
            }
        }

        echo "✅ {$numberOfPurchases} purchases with items & per-device inventory items created!\n";
    }
}
